/*
 * Create a wiring list corresponding to the map file
 */
#include <unistd.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>

#include "enclosure.h"
#include "switchstuff.h"
#include "map.h"
#include "gmstuff.h"

extern int My_host_id;
extern char *Wirelist;
extern int Hport;

int
find_the_packets(
  struct enclosure **pkt_ep,
  int *pkt_slot,
  int *pkt_port,
  int min,
  int max)
{
  int sw;
  int slot;
  int p;
  int diff;
  struct enclosure *ep;
  struct slotdesc *sp;
  struct xbarport *xp;
  struct enclosure *save_ep;
  int save_slot;
  int save_port;

  /* parse all enclosure files */
  parse_all_enclosures();
  
  save_ep = NULL;	/* no match found yet */

  for (sw=0; sw < Nswitch; ++sw) {
    ep = Switch + sw;
    if (ep->unmapped_slots == 0) continue;	/* skip known enclosures */

    for (slot=0; slot<MAX_SLOT; ++slot) {
      sp = ep->slot[slot];
      if (sp == NULL || (sp->xbar != NULL)) continue;

      for (p=0; p<sp->num_xp; ++p) {
        xp = sp->xp + p;
        diff = xp->vals.invalidroutes - xp->old_vals.invalidroutes;
        if (diff > 0) {
          printf("%s, slot %d, port %d, diff=%d, o=%d, n=%d\n",
		ep->name, slot, p, diff,
		xp->old_vals.invalidroutes,
		xp->vals.invalidroutes);
        }
        if (diff >= min && diff <= max) {
          printf("I am attached to %s, slot %d, port %d\n",
		ep->name, slot, p);
	  if (save_ep == NULL) {
	    save_ep = ep;
	    save_slot = slot;
	    save_port = p;
          } else {
	    fprintf(stderr, "Multiple invalidroute matches\n");
	    return 0;
          }
        }
      }
    }
  }

  if (save_ep != NULL) {
    *pkt_ep = save_ep;
    *pkt_slot = save_slot;
    *pkt_port = save_port;
    return 1;
  }

  fprintf(stderr, "Ack! Cannot find packets in enclosure!\n");
  return 0;
}

int
create_wiring_list(struct mapfile *mp)
{
  struct enclosure *ep;
  struct enclosure *nep;
  struct slotdesc *sp;
  struct slotdesc *sp2;
  struct fiberport *fp;
  struct xbarport *xp;
  struct xbar *xbar;
  struct xbar *xbar2;
  int xport;
  int xport2;
  int s;
  int slot;
  int p;
  int rc;
  int nslot;
  int nport;
  int retry;

  /* get the xbar attached to our host */
  xbar = mp->host[My_host_id].port[Hport].xbar;
  
  for (retry=0; retry<30; ++retry) {
    send_invalid_route(xbar->route, xbar->rlen, 20, 20*2*1024);
    sleep(1);
    rc = find_the_packets(&nep, &nslot, &nport, 30, 42);
    if (rc > 0) {
      break;
    } else {
      printf("Retrying find...\n");
    }
  }

  /* If we never made it, just stop */
  if (rc == 0) {
    printf("Cannot find packets, giving up.\n");
    printf("WARNING - wire list will be only partial!\n");
    return -1;
  }

  /* Associate this host with its fiber port */
  associate_enclosure(nep, nslot, nport, xbar,
	mp->host[My_host_id].port[Hport].xbport);

  /*
   * Now, loop through all switches/slots/ports scouting unknown frontiers
   */
 restart:
  for (s=0; s<Nswitch; ++s) {
    ep = Switch + s;
    
    /* Find a fiber port that goes nowhere... */
    for (slot=1; slot<MAX_SLOT; ++slot) {
      sp = ep->slot[slot];
      if (sp == NULL || sp->num_fiber == 0) {
        continue;
      }

      for (p=0; p<sp->num_fiber; ++p) {
        fp = sp->fp + p;
        xp = fp->xp;
	sp2 = xp->ep->slot[xp->slotno];

	/* We need to find a fiber port whose underlying xbar known */
	xbar = sp2->xbar;
	if (xbar == NULL) {
	  continue;
	}
	xport = WPORT_TO_XPORT(xp->portno, sp2->port_delta);

	if (xbar->port[xport].conn_type != CONN_XBAR) {
	  continue;
	}

	/* OK, finally, this is the xbar on the other end of the fiber */
	xbar2 = xbar->port[xport].ptr.x;

	/*
	 * The test packet did not necessarily come in the same way we found
	 * the xbar, but we happened to save port the packet entered thru
	 * back when we were generating the routes.  Hence, we set xport2 as
	 * below instead of xport2 = xbar->port[xport].conn_port
	 */
	xport2 = xbar2->in_port;

	/* If not already mapped, find it! */
	if (xbar2->sp == NULL) {

	  for (retry=0; retry<3; ++retry) {
	    send_invalid_route(xbar2->route, xbar2->rlen, 20, 20*2*1024);
	    sleep(1);
	    rc = find_the_packets(&nep, &nslot, &nport, 30, 42);
	    if (rc > 0) {
	      break;
	    } else {
	      printf("Retrying find...\n");
	    }
	  }

	  /* If we never made it, just stop */
	  if (rc == 0) {
	    printf("Cannot find packets, giving up.\n");
	    printf("WARNING - wire list will be only partial!\n");
	    return -1;
	  }

	  /* Associate this new enclosure */
	  associate_enclosure(nep, nslot, nport, xbar2, xport2);
	  goto restart;
	}
      }
    }
  }

  printf("All done with enclosure mapping\n");
  return 0;
}

	  
void
output_wiring_list()
{
  struct enclosure *ep;
  struct slotdesc *sp;
  struct slotdesc *sp2;
  struct slotdesc *sp3;
  int e;
  int slot;
  int port;
	int hport;
  int mxp;
  struct fiberport *fp;
  struct fiberport *fp2;
  struct xbarport *xp2;
  struct xbar *xbar;
  FILE *f;

  if (Wirelist == NULL || strcmp(Wirelist, "-") == 0) {
    printf("Writing wirelist to stdout\n");
    f = stdout;
  } else {
    printf("Writing wirelist to %s\n", Wirelist);
    f = fopen(Wirelist, "w");
    assert(f != NULL);
  }

  for (e=0; e<Nswitch; ++e) {
    ep = Switch + e;

    for (slot=1; slot < MAX_SLOT; ++slot) {
      sp = ep->slot[slot];

      /* skip missing slots */
      if (sp == NULL) continue;

      fprintf(f, "\nEnclosure %s Slot %d serial %s model %s",
	ep->name, slot, sp->serialno, sp->model);
      if (sp->num_xp > 0) {
	fprintf(f, " %s", sp->xbar->name);
      }
      fprintf(f, "\n");

      for (port = 0; port < sp->num_fiber; ++port) {

	fp = sp->fp + port;
	fprintf(f, "Port %d ", port + sp->port_label_offset);

	/* Is our xbar known? */
	if (fp->xp == NULL || ep->slot[fp->xp->slotno]->xbar == NULL) {
	  fprintf(f, "????\n");

	} else {
	  sp2 = ep->slot[fp->xp->slotno];
	  xbar = sp2->xbar;
	  mxp = WPORT_TO_XPORT(fp->xp->portno, sp2->port_delta);

	  /* print the internal xbar:port for this port */
	  fprintf(f, "(%s:%d) ",
		  fp->xp->ep->slot[fp->xp->slotno]->xbar->name,
		  fp->xp->portno);

	  switch (xbar->port[mxp].conn_type) {
	  case CONN_NULL:
	    fprintf(f, "nothing\n");
	    break;
	  case CONN_HOST:
	    fprintf(f, "host %s, port ", xbar->port[mxp].ptr.hp->myhost->hostname);

		/* find the host port the xbar is connected to */
		for (hport = 0; hport < xbar->port[mxp].ptr.hp->myhost->nport; hport++) {
			if (&(xbar->port[mxp].ptr.hp->myhost->port[hport]) == xbar->port[mxp].ptr.hp) {
				fprintf(f, "%d\n", hport);
				break;
			}
		}
		assert(hport < xbar->port[mxp].ptr.hp->myhost->nport);
	    break;
	  case CONN_XBAR:
	    /* slot holding remote xbar */
	    sp3 = xbar->port[mxp].ptr.x->sp;

	    /* NULL sp3 means we never translated this xbar */
	    if (sp3 == NULL) {
	      fprintf(f, "untranslated_xbar\n");
	      break;
	    }

	    /* xbar port on that slot */
	    xp2 = sp3->xp + XPORT_TO_WPORT(xbar->port[mxp].conn_port,
			sp3->port_delta);

	    /* fiber port for that xbar port */
	    fp2 = xp2->fp;

	    /* print info about this fiber port */
	    fprintf(f, "switch %s slot %d port %d\n",
		  fp2->ep->name, fp2->slotno,
		  fp2->portno + fp2->ep->slot[fp2->slotno]->port_label_offset);
	    break;
	  }
	}
      }
    }
  }

  if (f != stdout) {
    fclose(f);
  }   
}
